/************************************************************************
 * NAME:	dir
 *
 * DESCR:	Directory functions.
 *
 * NOTES:	
 ************************************************************************/
#include "coco.h"
#include "../fs-utility.h"


/************************************************************************
 * NAME:	coco_dirent_unpack() & _pack()
 *
 * DESCR:	Un/packs a directory entry in the given buffer, and fills
 *		in a dirent structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
coco_dirent_unpack(struct coco_dirent *entry, unsigned char *buffer)
{
    memcpy(entry->name,&buffer[0],8);
    memcpy(entry->ext,&buffer[8],3);
    entry->filetype = buffer[11];
    entry->ascii = buffer[12];
    entry->firstgran = buffer[13];
    entry->lastsecbytes = coco_int_unpack(&buffer[14]);
    memcpy(entry->reserved,&buffer[16],16);

    return(TRUE);
}
 
static int
coco_dirent_pack(struct coco_dirent *entry, unsigned char *buffer)
{
    memcpy(&buffer[0],entry->name,8);
    memcpy(&buffer[8],entry->ext,3);
    buffer[11] = entry->filetype & 0x00ff;
    buffer[12] = entry->ascii & 0x00ff;
    buffer[13] = entry->firstgran & 0x00ff;
    coco_int_pack(entry->lastsecbytes,&buffer[14]);
    memcpy(&buffer[16],entry->reserved,16);

    return(TRUE);
}

/************************************************************************
 * NAME:	coco_dir_read() & coco_dir_write()
 *
 * DESCR:	Reads the directory on the current disk, and stores it in
 *		the directory buffer.  Remember to write it back out if
 *		it changes! (ie - a file is changed/deleted/added)
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the directory was read alright...FALSE otherwise.
 *
 * NOTES:
 ************************************************************************/
int
coco_dir_read(struct cocofs	*cocofs)
{
    struct coco_dir	*cdir = &cocofs->dirbuf;
    struct coco_dirent	*entry;
    int			 sector = COCO_DIR_SECTOR_START;
    char		 buffer[256];
    int			 i,j;

    for ( i=0; i < COCO_DIR_SECTORS; i++) {
	if (!coco_getpsect(cocofs,COCO_DIR_TRACK,sector+i,buffer)) {
	    return(FALSE);
	}

	for (j=0; j < COCO_ENTRY_PER_SECTOR; j++) {
	    entry = &cdir->dirent[i*COCO_ENTRY_PER_SECTOR+j];
	    if (!coco_dirent_unpack(entry,buffer+(COCO_ENTRY_SIZE*j))) {
		return(FALSE);
	    }
	}
    }

    return(TRUE);
}
int
coco_dir_write(struct cocofs	*cocofs)
{
    struct coco_dir	*cdir = &cocofs->dirbuf;
    struct coco_dirent	*entry;
    int			 sector = COCO_DIR_SECTOR_START;
    char		 buffer[256];
    int			 i,j;

    for ( i=0; i < COCO_DIR_SECTORS; i++) {
	for (j=0; j < COCO_ENTRY_PER_SECTOR; j++) {
	    entry = &cdir->dirent[i*COCO_ENTRY_PER_SECTOR+j];
	    if (!coco_dirent_pack(entry,buffer+(COCO_ENTRY_SIZE*j))) {
		return(FALSE);
	    }
	}
	if (!coco_putpsect(cocofs,COCO_DIR_TRACK,sector+i,buffer)) {
	    return(FALSE);
	}
    }

    return(TRUE);
}
    
/************************************************************************
 * NAME:	coco_name_pack() & coco_name_unpack()
 *
 * DESCR:	Pack the given name into a directory entry structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- base is assumed to be 8 chars and ext 3 chars
 *		- name is normally in LC and otherwise UC
 *		- name must be \0 terminated
 *		- it appears that in the directory, names are NOT space
 *		  padded, they are NULL padded.  For historical reasons,
 *		  the unpack will deal with space padding.
 ************************************************************************/
static void
coco_name_pack(char *name, char *base, char *ext)
{
    fs_filename_split(fs_basename(name), base, ext, FALSE);
}

static void
coco_name_unpack(char *name, char *base, char *ext)
{
    fs_filename_combine(name, base, ext);
}

/************************************************************************
 * NAME:	coco_dirent()
 *
 * DESCR:	Returns a pointer to the coco_dirent of the given inode.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
struct coco_dirent *
coco_dirent(struct cocofs *cocofs, coco_inode inode)
{
    struct coco_dir	*cdir = &cocofs->dirbuf;
    struct coco_dirent	*entry;
    int			sector_no = COCO_INODE_SECTOR(inode);
    int			entry_no = COCO_INODE_ENTRY(inode);

    entry = &cdir->dirent[sector_no*COCO_ENTRY_PER_SECTOR+entry_no];

    return(entry);
}

/************************************************************************
 * NAME:	coco_dir_filedel()
 *
 * DESCR:	Delete the given file from the disk, and return its
 *		groups to the free list (the RGT is updated for the
 *		heck of it, too).
 *
 * ARGS:	the cocofs_inode
 *
 * RETURNS:	TRUE if it was deleted, FALSE otherwise (file no exist)
 *
 * NOTES:	
 ************************************************************************/
void
coco_dir_filedel(struct cocofs *cocofs, coco_inode inode)
{
    struct coco_dirent *entry = coco_dirent(cocofs,inode);

    COCO_MARK_ENTRY_UNUSED(entry);
    coco_fat_free_chain(cocofs,entry->firstgran);
}

/************************************************************************
 * NAME:	coco_dir_update()
 *
 * DESCR:	Updates a particular directory entry with new information
 *		regarding groups in use, and number of sectors in the
 *		last group.  The modification date is updated too.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
coco_dir_update_entry(struct cocofs	*cocofs, 
		      coco_inode	inode,
		      int		firstgran,
		      int		curgran,
		      int		lastsectors,
		      int		lastbytes)
{
    struct coco_dirent	*entry = coco_dirent(cocofs,inode);

    entry->firstgran = firstgran;
    coco_fat_last_sectors(cocofs,curgran,lastsectors);
    entry->lastsecbytes = lastbytes;
}

/************************************************************************
 * NAME:	coco_dir_create()
 *
 * DESCR:	Create a new directory entry with the given parameters.
 *
 * ARGS:	
 *
 * RETURNS:	the inode of the new entry, INODE_NULL if none available.
 *
 * NOTES:	
 ************************************************************************/
coco_dir_create(struct cocofs *cocofs,
		char		*name,
		int		 filetype,
		int		 ascii_flag)
{
    struct coco_dirent	*entry;
    struct coco_dir	*cdir = &cocofs->dirbuf;
    int			i, j;
    char		filename[13];

    for ( i=0; i < COCO_DIR_SECTORS; i++) {
	for (j=0; j < COCO_ENTRY_PER_SECTOR; j++) {

	    entry = &cdir->dirent[i*COCO_ENTRY_PER_SECTOR+j];

	    if (COCO_ENTRY_UNUSED(entry)) {
		goto found;		/* sorry for the goto!	*/
	    }

	    if (COCO_ENTRY_UNUSED_LAST(entry)) {
		goto found;		/* sorry for the goto!	*/
	    }
	}
    }

 found:
    if ( i == COCO_DIR_SECTORS ) {
	return(COCO_INODE_NULL);	/* no more free entries	*/
    }

    /* note that packing the name into the entry unmarks it as "unused"	*/

    coco_name_pack(name, entry->name, entry->ext);

    entry->filetype = filetype;
    entry->ascii = ascii_flag;

    return(COCO_INODE(i,j));
}

/************************************************************************
 * NAME:	coco_dir_find()
 *
 * DESCR:	Finds the given filename in the directory.
 *
 * ARGS:	
 *
 * RETURNS:	The inode of the file if it is found, otherwise the NULL
 *		inode is returned.
 *
 * NOTES:	
 ************************************************************************/
coco_inode
coco_dir_find(struct cocofs *cocofs, char *filename)
{
    int			i, j;
    char		target[13];
    struct coco_dir	*cdir = &cocofs->dirbuf;
    struct coco_dirent	*entry;

    for ( i=0; i < COCO_DIR_SECTORS; i++) {
	for (j=0; j < COCO_ENTRY_PER_SECTOR; j++) {

	    entry = &cdir->dirent[i*COCO_ENTRY_PER_SECTOR+j];

	    if (!COCO_ENTRY_UNUSED(entry) && !COCO_ENTRY_UNUSED_LAST(entry)) {

		coco_name_unpack(target,entry->name,entry->ext);

		if (strcmp(target,filename) == 0) {
		    return(COCO_INODE(i,j));
		}
	    }
	}
    }
    return(COCO_INODE_NULL);
}


/************************************************************************
 * NAME:	coco_dir_report()
 *
 * DESCR:	Generate a report of the current directory.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
coco_dir_report(struct cocofs *cocofs, int verbosity)
{
    struct coco_dir	*cdir = &cocofs->dirbuf;
    struct coco_dirent	*entry;
    int			i, j;
    char		filename[13];
    int			grans,sectors;
    int			totalsize;

    if (verbosity == 2) {
    }

    printf("Filename      bytes  sectors grans type\n");
    printf("------------ ------- ------- ----- ----\n");

    for ( i=0; i < COCO_DIR_SECTORS; i++) {
	for (j=0; j < COCO_ENTRY_PER_SECTOR; j++) {
	    entry = &cdir->dirent[i*COCO_ENTRY_PER_SECTOR+j];

	    if (!COCO_ENTRY_UNUSED(entry) && !COCO_ENTRY_UNUSED_LAST(entry)) {
		int	groups;
		int	size;

		coco_name_unpack(filename,entry->name,entry->ext);

		sectors = coco_dir_file_size(cocofs,entry);
		grans = (sectors+COCO_SECTORS_PER_GRAN-1) / COCO_SECTORS_PER_GRAN;

		printf("%-12.12s ",filename);
/*		printf("%6dk   %2d", sectors*COCO_SECTOR_SIZE/1024,grans);	*/
		printf("%6d     %3d   %2d   %1d %2d", (sectors-1)*COCO_SECTOR_SIZE+entry->lastsecbytes,sectors,grans,entry->filetype,entry->ascii);
		printf("\n");
	    }
	}
    }

    {
	int bytesfree = coco_fat_free_grans(cocofs)*COCO_SECTORS_PER_GRAN*COCO_SECTOR_SIZE;

	printf( "Free space: %d gran%s (%s%dk)\n", 
			  coco_fat_free_grans(cocofs),
			  (coco_fat_free_grans(cocofs)==1)?"":"s",
			  (bytesfree < 1024 && bytesfree!= 0)?".":"",
			  (bytesfree < 1024)?bytesfree/100:bytesfree/1024);
    }
}

/************************************************************************
 * NAME:	coco_dir_file_size()
 *
 * DESCR:	Returns the size of the given directory entry in sectors.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
coco_dir_file_size(struct cocofs *cocofs, struct coco_dirent *entry)
{
    int	sectors;
    int gran_count = 0;
    int	g;

    for (g = entry->firstgran; !COCO_FAT_LAST_GRAN(cocofs,g); g = COCO_FAT_NEXT_GRAN(cocofs,g)) {
	gran_count++;
    }

    sectors = (gran_count * COCO_SECTORS_PER_GRAN) + COCO_FAT_LAST_SECTORS(cocofs,g);
    
    return(sectors);
}

/************************************************************************
 * NAME:	coco_dir_init()
 *
 * DESCR:	Initialize the given directory.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
coco_dir_init(struct coco_dir *cdir)
{
}

void
coco_dir_cleanup(struct coco_dir *hdir)
{
}

